#!/usr/bin/python
# -*- coding: utf-8 -*-

# --------------------------------------------------------------- #
#    Library for a simple processing of ASF files.                #
# --------------------------------------------------------------- #
#    2011 - 2016  Zdenek Hanzlicek (zhanzlic@ntis.zcu.cz)         #
#                 NTIS, University of West Bohemia                #
# --------------------------------------------------------------- #

# SVN $Id: asflight.py 1756 2016-04-15 13:39:57Z zhanzlic $


import codecs
import os
import os.path


class AsfLight:

    def __init__( self, file_name=None, code_page='utf-8' ):

        self.utts = dict()  # particular utterances
        self.attrib_order = list()  # order of attributes for all utterances
        self.header = list()  # list of lines with comments

        if file_name is not None:
            self.read( file_name )


    ## ==========----------


    def __len__( self ):
        return len( self.utts )


    ## ==========----------


    def __iter__( self ):
        return iter( self.utts )


    ## ==========----------


    def __setitem__( self, key, value ):
        self.utts[ key ] = value


    ## ==========----------


    def __getitem__( self, key ):
        return self.utts[ key ]


    ## ==========----------


    def __contains__( self, item ):
        return item in self.utts


    ## ==========----------


    def read( self, file_name, code_page='utf-8' ):

        asf_handle = codecs.open( file_name, 'rt', code_page )
        asf_content = asf_handle.readlines()
        asf_handle.close()

        for asf_line in asf_content:
            asf_line = asf_line.strip()

            if asf_line == "":  # empty line
                continue

            if asf_line.startswith("#"):  # header / comment
                self.header.append( asf_line )

            elif asf_line.startswith('"') and asf_line.endswith('"'):  # new utterance
                utt_name = asf_line[1:-1]
                utt_units = list()
                self.utts[ utt_name ] = utt_units

            elif asf_line.startswith("|") and asf_line.endswith('|'):  # unit
                attrib_vals = [ attrib_val.strip() for attrib_val in asf_line[1:-1].split("|") ]
                utt_units.append( { self.attrib_order[ idx ]:attrib_vals[ idx ] for idx in range( len( self.attrib_order ) ) } )

            elif asf_line.startswith("[") and asf_line.endswith("]"):  # list of attribute names
                self.attrib_order = [ attrib_name.strip() for attrib_name in asf_line[1:-1].split("|") ]


    ## ==========----------


    def write( self, file_name, code_page='utf-8' ):

        asf_handle = codecs.open( file_name, 'wt', code_page )

        if len( self.header ):
            asf_handle.write( "\n".join( self.header ) )
            asf_handle.write( "\n\n" )

        attrib_lens = { attrib_name:len( attrib_name ) for attrib_name in self.attrib_order }

        # get the maximum lenghts for particular attributes
        for units in self.utts.itervalues():
            for unit in units:
                for attrib_name in unit:

                    attrib_val = unit[ attrib_name ]
                    if not isinstance( attrib_val, unicode ):
                        attrib_len = len( unicode( attrib_val ) )
                    else:
                        attrib_len = len( attrib_val )

                    if attrib_lens[ attrib_name ] < attrib_len:
                        attrib_lens[ attrib_name ] = attrib_len

        # write list of attribute names
        asf_handle.write( "[ " + " | ".join( [ attrib_name + " "*( attrib_lens[ attrib_name ] - len( attrib_name ) ) for attrib_name in self.attrib_order ] ) + " ]\n\n" )

        for utt_name in sorted( self.utts.iterkeys() ):
            asf_handle.write( '"' + utt_name + '"\n' )

            for unit in self.utts[ utt_name ]:
                attrib_vals = { attrib_name:( unit[ attrib_name ] if isinstance( unit[ attrib_name ], unicode ) else unicode( unit[ attrib_name ] ) ) for attrib_name in self.attrib_order }
                asf_handle.write( "| " + " | ".join( [ attrib_vals[ attrib_name ] + " "*( attrib_lens[ attrib_name ] - len( attrib_vals[ attrib_name ] ) ) for attrib_name in self.attrib_order ] ) + " |\n" )

            asf_handle.write( "\n" )

        asf_handle.close()

